/*******************************************************************************
 * WeaponInvMindClaw.uc
 * The Mind Claw inventory item
 * GreatEmerald, 2012
 ******************************************************************************/  
//=============================================================================
// $Workfile: WeaponInvMindclaw.uc $
// Created By: Mark Poesch
// Created On: 9/28/00
// $Author: Sbrown $
// $Date: 5/10/01 10:43p $
// $Revision: 6 $
//=============================================================================
class WeaponInvMindClaw extends U2Weapon;

#exec OBJ LOAD File=..\Meshes\GlmWeaponsG.ugx
#exec OBJ LOAD File=..\Textures\GlmWeaponsT.utx


var() float AttackViewAngle;		// 1.0 = directly in front, 0.707 = 90 degree cone, 0.5 = 120 degree cone, 0.0 = 180 degrees
var() float MaxSightRadius;			//
var() float MinAttackTime;			//
var() float DamagePerSecond;		//
var() float AbsorbDamageRatio;		//
var() class<PulseLineGenerator> AttackEffectClass;

var byte bAttacking[16];			//
var Actor Attacking[16];			//
var float AttackTime[16];			//
var PulseLineGenerator AttackEffect[16];


simulated function ClearWeaponEffects()
{
	local int i;

	Pawn(Owner).PlaySound( None, SLOT_Misc );

	for( i = 0; i < ArrayCount(AttackEffect); i++ )
	{
		if( AttackEffect[i] != None )
		{
			AttackEffect[i].Destroy();
			AttackEffect[i] = None;
		}
	}

	Super.ClearWeaponEffects();
}


state Firing
{
	simulated event BeginState()
	{
		Pawn(Owner).PlaySound( sound'U2WeaponsA.SkaarjGlove.SG_HeavyFire', SLOT_Misc, Pawn(Owner).SoundDampening );
	}

	simulated event EndState()
	{
		Pawn(Owner).PlaySound( None, SLOT_Misc );
	}

	function bool InvalidTarget( Actor A )
	{
		local float InView;

		// not visible to us?
		if( A == Owner || A.bHidden || !FastTrace( A.Location, Owner.Location ) )
			return false;

		// outside our field of view?
		InView = Normal( A.Location - Owner.Location ) dot vector( Pawn(Owner).GetViewRotation() );
		if( InView < AttackViewAngle )
			return false;

		// is it already dead?  is it a sentient (human) target?
		if( Pawn(A) != None && ( Pawn(A).Health <= 0 || !Pawn(A).IsPlayer() ) ) //MDF is this sufficient?
			return false;

		return true;
	}

	function UpdateAttackList( float DeltaTime )
	{
		local Pawn P;
		local int i;

		for( i = 0; i < ArrayCount(bAttacking); i++ )
		{
			bAttacking[i] = 0;
		}

		foreach RadiusActors( class'Pawn', P, MaxSightRadius )
		{
			if( InvalidTarget( P ) )
			{
				// if target is in Attacking list, continue tracking them by setting bAttacking
				for( i = 0; i < ArrayCount(Attacking); i++ )
				{
					if( P == Attacking[i] )
					{
						bAttacking[i] = 1;
						break;
					}
				}

				// if not in the Attacking list, then add them
				if( i >= ArrayCount(Attacking) )
				{
					for( i = 0; i < ArrayCount(Attacking); i++ )
					{
						if( Attacking[i] == None )
						{
							bAttacking[i] = 1;
							Attacking[i] = P;
							break;
						}
					}
				}
			}
		}

		// accumulate time for current targets, and clear time for dropped targets, and
		// update AttackingEffect (destroy or create as appropriate)
		for( i = 0; i < ArrayCount(bAttacking); i++ )
		{
			if( bAttacking[i] == 0 )
			{
				Attacking[i] = None;
				AttackTime[i] = 0.0;

				if( AttackEffect[i] != None )
				{
					AttackEffect[i].Destroy();
					AttackEffect[i] = None;
				}
			}
			else
			{
				AttackTime[i] += DeltaTime;

				if( AttackEffect[i] == None )
				{
					AttackEffect[i] = Spawn( AttackEffectClass );
					AttackEffect[i].AddConnection(Owner.Location, Attacking[i].Location);
				}
				AttackEffect[i].BeamWidth.A = AttackTime[i] * 1;
				AttackEffect[i].BeamWidth.B = AttackTime[i] * 4;
			}
		}
	}

	function bool Attack()
	{
		local int i;
		local float Damage;
		local bool bAttacked;

		for( i = 0; i < ArrayCount(Attacking); i++ )
		{
			if( Attacking[i] != None && AttackTime[i] > MinAttackTime )
			{
				bAttacked = true;

				// inflict damage
				Damage = DamagePerSecond * AttackTime[i];
				Attacking[i].TakeDamage( Damage, Instigator, Attacking[i].Location, Attacking[i].Location - Pawn(Owner).Location, MyDamageType );

				// absorb some of the damage inflicted, capped at 199
				Pawn(Owner).Health += Damage * AbsorbDamageRatio;
				Pawn(Owner).Health = Min( Pawn(Owner).Health, 199 );
			}

			bAttacking[i] = 0;
			Attacking[i] = None;
			AttackTime[i] = 0.0;
			if( AttackEffect[i] != None )
			{
				AttackEffect[i].Destroy();
				AttackEffect[i] = None;
			}
		}

		return bAttacked;
	}

	simulated event Tick( float DeltaTime )
	{
		UpdateAttackList( DeltaTime );

		if( Pawn(Owner).Controller.bFire == 0 )
		{
			if( Attack() )
			{
				Pawn(Owner).PlaySound( sound'U2WeaponsA.SkaarjGlove.SG_HeavyFire', SLOT_None, Pawn(Owner).SoundDampening );
			}
			else
			{
				Pawn(Owner).PlaySound( sound'U2XMPA.EnergyRelay.EnergyRelayShutdown', SLOT_None, Pawn(Owner).SoundDampening );
			}
			Finish();
		}
	}

	simulated event AnimEnd(int Channel)
	{
		// don't call Finish() when anim finishes
	}
}


state ClientFiring
{
	simulated event Tick( float DeltaTime )
	{
		if( Pawn(Owner).Controller.bFire == 0 )
		{
			Finish();
		}
	}

	simulated event AnimEnd(int Channel)
	{
		// don't call Finish() when anim finishes
	}
}


simulated function AltFire(  )
{
	//ARL activate Anti-Magic Aura effect around player
}

defaultproperties
{
	AttackViewAngle=0.500000
	MaxSightRadius=1024.000000
	MinAttackTime=1.000000
	DamagePerSecond=100.000000
	AbsorbDamageRatio=0.200000
	AttackEffectClass=Class'ParticleSystems.PulseLineGenerator'
	WeaponAnimationType=AT_Large
	MomentumTransfer=40000.000000
	FireAnimRate=0.400000
	AltFireAnimRate=0.400000
	AltShakeMag=0.000000
	AltShakeTime=0.000000
	AltShakeVert=(Z=0.000000)
	WeaponKickDirection=(X=0.000000)
	WeaponKickDuration=0.000000
	WeaponKickRecoveryDuration=0.000000
	AltWeaponKickDirection=(X=0.000000)
	AltWeaponKickDuration=0.000000
	AltWeaponKickRecoveryDuration=0.000000
	//AmmoName=Class'U2Weapons.AmmoInvPsyCrystal'
	AmmoName=Class'U2Weapons.ammoInvAssaultRifle'
	PickupAmmoCount=50
	AutoSwitchPriority=6
	MyDamageType=Class'U2.DamageTypePsionic'
	ShakeMag=0.000000
	shaketime=0.000000
	ShakeVert=(Z=0.000000)
	AIRating=0.700000
	RefireRate=0.400000
	AltRefireRate=0.400000
	//AltFireSound=Sound'U2Weapons.MindClaw.MC_AltFire'
	//CockingSound=Sound'U2Weapons.MindClaw.MC_Reload'
	//SelectSound=Sound'U2Weapons.MindClaw.MC_Select'
	AltFireSound=Sound'U2WeaponsA.SkaarjGlove.SG_AltFire'
	CockingSound=Sound'U2WeaponsA.LaserRifle.LR_Reload'
	SelectSound=Sound'U2WeaponsA.SkaarjGlove.SG_Select'
	InventoryGroup=3
	PickupClass=Class'MindClaw.weaponMindClaw'
	PlayerViewOffset=(X=40.599998,Y=14.600000,Z=-16.600000)
	//FirstPersonMesh=VertMesh'U2Weapons.MC_FP'
	//StatusIcon=Texture'U2Weapons.Icons.MC_Icon'
	ItemName="Mindclaw"
	//Mesh=VertMesh'U2Weapons.MC_TP'
	PickupAmmoCount=100
	FireSound=Sound'U2WeaponsA.SkaarjGlove.SG_HeavyFire'
	ThirdPersonMesh=LegendMesh'GlmWeaponsG.MC_TP'
	Mesh=LegendMesh'GlmWeaponsG.MC_FP'
	FireLastRoundSound=Sound'U2XMPA.EnergyRelay.EnergyRelayShutdown'
	AltFireLastRoundSound=Sound'U2XMPA.EnergyRelay.EnergyRelayShutdown'
	ReloadUnloadedSound=Sound'U2WeaponsA.LaserRifle.LR_ReloadUnloaded'
} 

/*defaultproperties
{
	FireTime=1.076923
	AltFireTime=0.875000
	FireLastDownTime=2.571428
	AltFireLastDownTime=1.500000
	FireLastReloadTime=2.562500
	AltFireLastReloadTime=2.562500
	ReloadTime=2.562500
     GunButtSounds(0)=Sound'U2WeaponsA.GunButt.GunButt01'
     GunButtSounds(1)=Sound'U2WeaponsA.GunButt.GunButt02'
     GunButtSounds(2)=Sound'U2WeaponsA.GunButt.GunButt03'
     GunButtSounds(3)=Sound'U2WeaponsA.GunButt.GunButt04'
     DecoEffects(0)=(AnimSequence=Fire,DecoClass=Class'ParticleSystems.ParticleSalamander',Particles=ParticleSalamander'ShotgunFX.ParticleSalamander0',MountNode="#Muzzleflash",bRequiresWorldZBuffer=true)
     DecoEffects(1)=(AnimSequence=Fire,DecoClass=Class'U2Weapons.S_LightEffect',MountNode="#Muzzleflash",MountOffset=(X=15.000000,Z=15.000000))
     DecoEffects(2)=(AnimSequence=AltFire,DecoClass=Class'ParticleSystems.ParticleSalamander',Particles=ParticleSalamander'ShotgunFX.ParticleSalamander0',MountNode="#Muzzleflash",bRequiresWorldZBuffer=true)
     DecoEffects(3)=(AnimSequence=AltFire,DecoClass=Class'U2Weapons.S_LightEffect',MountNode="#Muzzleflash",MountOffset=(X=15.000000,Z=15.000000))
     TargetableTypes(0)='Pawn'
     UseReticleOnEvents(0)="UseReticleText"
     UseReticleOnEvents(1)="UseReticleCorners"
     UseReticleOnEvents(2)="UseReticleTopBars"
     ProximityReticleOnEvents(0)="ProximityReticleCorners"
     ProximityReticleOnEvents(1)="ProximityReticleTopBars"
}*/
