/******************************************************************************
SBTriggerableSpawnPoint: Spawns the given *Prototype* at its location each time
this SpawnPoint is triggered, like a one-shot-ThingFactory and one SpawnPoint.
Its Event is fired on a successful spawn.
It also spawns things orientated according to its rotation, not only in
yaw-component of its rotation like a SpawnPoint.

Optionally there can be up to 16 properties assigned to the spawned Actor by
setting the property name and its value in array *Properties*.
Some special properties are supported:
- bBlockActors
- bBlockPlayers
- bCollideActors
- CollisionHeight
- CollisionRadius
- Disable
- Enable
- GotoState
- InitialState
- Log
- MultiSkins
- Physics
- RotationRate
- Velocity
- Warn

If a value begins with "MyLevel.", that part of string is replaced by the map
name.


Author: SeriousBarbie AT Barbies DOT World

Release Version 3: 18 Feb 2026
Name: SBTriggerableSpawnPointV3
* fixed a bug in setting *CollisionHeight*
+ added support for "MyLevel" in values

Release Version 2: 08 Feb 2026
Name: SBTriggerableSpawnPointV2
+ reduced Properties arrays from 32 to 16
+ added support for more special properties
* replaced warn(...) with Logger(LOG_Warning, ...)
+ added bTriggerOnceOnly
* changed two separate arrays for property/value to one

Release Version 1: 07 Dec 2025
Name: SBTriggerableSpawnPointV1
* expanded PrototypeProperties array lengths from 8 to 32
+ added support for arrays as PropertyValue
+ added support for array MultiSkins
+ added LogLevel for possibly debugging output

Release Version 0: 10 Jul 2023
Name: SBTriggerableSpawnPointV0
******************************************************************************/
class SBTriggerableSpawnPoint expands NavigationPoint;

#exec Texture Import File=Textures\SBTSP.pcx Name=SBTSP Mips=Off Flags=2

enum ELogType {
	LOG_None,
	LOG_Error,
	LOG_Warning,
	LOG_Info,
	LOG_Verbose,
	LOG_Debug,
	LOG_All,
};



var() struct TProperty {
	var() string Name;
	var() string Value;
} Properties[16];



struct TVector {
	var string Values[8];
	var byte Len;
};



var() class<actor> Prototype;
var() ELogType LogLevel;
var() bool bTriggerOnceOnly;

var name NameConversionHack;
var SBTriggerableSpawnPointInfo VersionInfos;


function bool CharIsDecDigit(byte CharCode) {
	return (CharCode >= 48 && CharCode <= 57);
}



function bool CharIsWhite(int charcode) {
/******************************************************************************
Returns TRUE if the char with *charcode* is a white char (= no ink is used if
printed).
******************************************************************************/

	switch (charcode) {
		case 32: // Space
			break;
		case 13: // CR
			break;
		case 10: // LF
			break;
		case 9: // tab
			break;
		case 12: // FF
			break;
		case 11: // vTab
			break;
		default:
			return false;
	}
	return true;
}



function bool IsNumericScalar(string s, optional out float f, optional out string Errormessage) {
/******************************************************************************
Detects the following cases as number and returns TRUE then:
[:blank:]*[+-]?[:digit:]+\.?[:digit:]*[:blank:]*

Special NONE VALID case: ".1"
******************************************************************************/
local int i;
//local bool bIsNegative, bHaveSign;
	Logger(LOG_All, "IsNumericScalar", "checking '" $ s $ "'");
	f = float(s);
	if (f != 0)
		goto IsNumeric_True;

	// if *f* is zero, *s* may be a representation of zero - or conversion to float failed

	// Let's do string processing then:
	s = Trim(s);
	// log("Checking '" $ s $ "'");
	if (s == "")
	{
		Errormessage = "empty value";
		goto IsNumeric_False;
	}
	// sign at first:
	if (Mid(s, 0, 1) == "-") i++;
	else if (Mid(s, 0, 1) == "+") i++;
/*
	if (Mid(s, i, 1) == ".")
	{
		warn("a number must have leading digits but s[" $ i $ "] is '" $ Mid(s, i, 1) $ "'");
		return false; // no leading digits -> invalid
	}
*/
	while (i < len(s) && CharIsDecDigit(Asc(Mid(s, i, 1))))
		i++;
	if (i >= len(s))
	{
		f = float(s);
		goto IsNumeric_True;
	}
	if (Mid(s, i, 1) == ".")
		i++;
	while (i < len(s) && CharIsDecDigit(Asc(Mid(s, i, 1))))
		i++;
	if (i >= len(s))
	{
		f = float(s);
		goto IsNumeric_True;
	}

IsNumeric_False:
	Errormessage = "'" $ s $ "' is not a valid number";
	Logger(LOG_Debug, "IsNumericScalar", Errormessage);
	return false;

IsNumeric_True:
	Logger(LOG_Debug, "IsNumericScalar", "'" $ s $ "' is numeric:" @ f);
	return true;

}



function bool IsNumericVector(TVector AVector, optional out int ErrorIndex) {
local int Result;

	//if (LogLevel >= LOG_All) LoggerDirect(LOG_All, "IsNumericVector", "function called with

	if (AVector.Len == 0)
	{
		ErrorIndex = -1;
		return false;
	}

	for (Result = 0; Result < AVector.Len; Result++)
		if ( ! IsNumericScalar(AVector.Values[Result]))
		{
			ErrorIndex = Result;
			return false;
		}
	return true;
}



event Logger(ELogType NeededLogLevel, coerce string FunctionName, coerce string msg) {
	if (LogLevel < NeededLogLevel)
		return;
	LoggerDirect(NeededLogLevel, FunctionName, msg);
}



event LoggerDirect(ELogType NeededLogLevel, coerce string FunctionName, coerce string msg) {
	msg = self $ "." $ FunctionName @ GetEnum(Enum'ELogType', NeededLogLevel) $ ":" @ msg;
	log(msg);
	BroadcastMessage(msg);
}



event PostBeginPlay() {

	Foreach AllActors(class'SBTriggerableSpawnPointInfo', VersionInfos)
		break;
	if (VersionInfos == None)
	{
		VersionInfos = spawn(class'SBTriggerableSpawnPointInfo');
		Logger(LOG_Info, "PostBeginPlay", VersionInfos.CProjectName  @ "by SeriousBarbie@Barbies.World, version" @ VersionInfos.CProjectVersion $ ", release" @ VersionInfos.CProjectReleasedate);
	}
	Super.PostBeginPlay();
}



function int SetProperties(Actor A, out int ChangedCount) {
// returns TRUE if change was successful
local int i;
local float f;
local int ErrorCount;
local string s;

	for (i = 0; i < ArrayCount(Properties); i++)
		if (Properties[i].Name != "")
		{
			// replace "MyLevel" with map name:
			if (StringStartsWithMyLevel(Properties[i].Value, s))
			{
				if (LogLevel >= LOG_Debug) Logger(LOG_Debug, "ChangeProperties", "about to change '" $ Properties[i].Value $ "' for" @ A $ "." $ Properties[i].Name @ "to '" $ s $ "'");
				Properties[i].Value = s;
			}
			if (SpecialPropertyHandling(A, Properties[i]))
				goto NextLoop;

			Logger(LOG_Debug, "ChangeProperties", "about to set '" $ Properties[i].Value $ "' for" @ A $ "." $ Properties[i].Name);
			A.SetPropertyText(Properties[i].Name, Properties[i].Value);
			if (IsNumericScalar(Properties[i].Value, f))
			{
				if (float(A.GetPropertyText(Properties[i].Name)) == f)
					goto Change_Success;
				else
					goto Change_Failed;
			}
			else
				if ((A.GetPropertyText(Properties[i].Name) ~= Properties[i].Value))
					goto Change_Success;
				else
				{
					if (A.GetPropertyText(Properties[i].Name) ~= "None" && Properties[i].Value == "") // special equal case: "None == ''"
						goto Change_Success;
					else
						goto Change_Failed;
				}
		Change_Success:
			Logger(LOG_Debug, "ChangeProperties", "successfully set" @ A $ "." $ Properties[i].Name $ "=" $ A.GetPropertyText(Properties[i].Name));
			ChangedCount++;
			goto NextLoop;

		Change_Failed:
			ErrorCount++;
			Logger(LOG_Warning, "ChangeProperties", "Setting property '" $ Properties[i].Name $ "' with value '" $ Properties[i].Value $ "' failed, current value=" $ A.GetPropertyText(Properties[i].Name));
			goto NextLoop;

		NextLoop:
		}
	return ErrorCount;
}



function bool SetPropertyPhysics(Actor A, string NewPhysics) {
	if (NewPhysics ~= "PHYS_None")
	{
		A.SetPhysics(PHYS_None);
		return A.Physics == PHYS_None;
	}
	else if (NewPhysics ~= "PHYS_Walking")
	{
		A.SetPhysics(PHYS_Walking);
		return A.Physics == PHYS_Walking;
	}
	else if (NewPhysics ~= "PHYS_Falling")
	{
		A.SetPhysics(PHYS_Falling);
		return A.Physics == PHYS_Falling;
	}
	else if (NewPhysics ~= "PHYS_Swimming")
	{
		A.SetPhysics(PHYS_Swimming);
		return A.Physics == PHYS_Swimming;
	}
	else if (NewPhysics ~= "PHYS_Flying")
	{
		A.SetPhysics(PHYS_Flying);
		return A.Physics == PHYS_Flying;
	}
	else if (NewPhysics ~= "PHYS_Rotating")
	{
		A.SetPhysics(PHYS_Rotating);
		return A.Physics == PHYS_Rotating;
	}
	else if (NewPhysics ~= "PHYS_Projectile")
	{
		A.SetPhysics(PHYS_Projectile);
		return A.Physics == PHYS_Projectile;
	}
	else if (NewPhysics ~= "PHYS_Rolling")
	{
		A.SetPhysics(PHYS_Rolling);
		return A.Physics == PHYS_Rolling;
	}
	else if (NewPhysics ~= "PHYS_Interpolating")
	{
		A.SetPhysics(PHYS_Interpolating);
		return A.Physics == PHYS_Interpolating;
	}
	else if (NewPhysics ~= "PHYS_MovingBrush")
	{
		A.SetPhysics(PHYS_MovingBrush);
		return A.Physics == PHYS_MovingBrush;
	}
	else if (NewPhysics ~= "PHYS_Spider")
	{
		A.SetPhysics(PHYS_Spider);
		return A.Physics == PHYS_Spider;
	}
	else if (NewPhysics ~= "PHYS_Trailer")
	{
		A.SetPhysics(PHYS_Trailer);
		return A.Physics == PHYS_Trailer;
	}
	else return false;
}



function bool SpecialPropertyHandling(Actor A, TProperty Property) {
// Returns TRUE if a special property name was detected.
local string ArrayName, ErrorMessage, SuccessMessage;
local int ArrayIndex;
local Texture tex;
local TVector AVector;
local vector V;


	if (StringIsArray(Property.Name, ArrayName, ArrayIndex))
	{
		if (ArrayName ~= "MultiSkins")
		{
			tex = Texture(DynamicLoadObject(Property.Value, class'Texture'));
			if (Tex != None)
				A.MultiSkins[ArrayIndex] = tex;
			else
			{
				ErrorMessage = "could not load Texture '" $ Property.Value $ "'";
				goto Change_Failed;
			}
			goto Change_Success;
		}
	}
	if (StringIsVector(Property.Value, AVector, ErrorMessage))
	{
		if (ErrorMessage != "") // *Property.Value* is an array but has invalid content
			return true;

		if (Property.Name ~= "Velocity")
		{
			if (AVector.Len != 3 || ! IsNumericVector(AVector))
			{
				ErrorMessage = "the value for property '" $ Property.Name $ "' must be a vector with 3 numeric components";
				goto Change_Failed;
			}
			V = TVectorToVector(AVector);
			A.Velocity = V;
			log("A.Velocity=" $ A.Velocity);
			log("V=" $ v);
			if (A.Velocity != V) goto Change_Failed;
			goto Change_Success;
		}

		if (Property.Name ~= "Rotation")
		{
			if (AVector.Len != 3 || ! IsNumericVector(AVector))
			{
				ErrorMessage = "the value for property '" $ Property.Name $ "' must be a vector with 3 numeric components";
				goto Change_Failed;
			}
			V = TVectorToVector(AVector);
			A.SetRotation(rotator(V));
			if (A.Rotation != rotator(V)) goto Change_Failed;
			goto Change_Success;
		}

		if (Property.Name ~= "RotationRate")
		{
			if (AVector.Len != 3 || ! IsNumericVector(AVector))
			{
				ErrorMessage = "the value for property '" $ Property.Name $ "' must be a vector with 3 numeric components";
				goto Change_Failed;
			}
			V = TVectorToVector(AVector);
			A.RotationRate = rotator(V);
			if (A.RotationRate != rotator(V)) goto Change_Failed;
			goto Change_Success;
		}
	}
	if (Property.Name ~= "CollisionRadius")
	{
		A.SetCollisionSize(float(Property.Value), A.CollisionHeight);
		if (A.CollisionRadius == float(Property.Value))
			goto Change_Success;
		else
			goto Change_Failed;
	}
	if (Property.Name ~= "CollisionHeight")
	{
		A.SetCollisionSize(A.CollisionRadius, float(Property.Value));
		if (A.CollisionHeight == float(Property.Value))
			goto Change_Success;
		else
			goto Change_Failed;
	}
	if (Property.Name ~= "bCollideActors")
	{
		A.SetCollision(bool(Property.Value));
		if (A.bCollideActors != bool(Property.Value))
			goto Change_Success;
		else
			goto Change_Failed;
	}
	if (Property.Name ~= "bBlockActors")
	{
		A.SetCollision(, bool(Property.Value));
		if (A.bBlockActors != bool(Property.Value))
			goto Change_Success;
		else
			goto Change_Failed;
	}
	if (Property.Name ~= "bBlockPlayers")
	{
		A.SetCollision(, , bool(Property.Value));
		if (A.bBlockPlayers != bool(Property.Value))
			goto Change_Success;
		else
			goto Change_Failed;
	}
	if (Property.Name ~= "InitialState")
	{
		A.InitialState = StringToName(Property.Value);
		A.GotoState(A.InitialState);
		if (string(A.GetStateName()) ~= Property.Value)
		{
			goto Change_Success;
			//Logger(LOG_Debug, "SpecialPropertyHandling", A $ ".InitialState successfully set to '" $ string(A.InitialState) $ "'");
		}
		else
		{
			goto Change_Failed;
			//Logger(LOG_Debug, "SpecialPropertyHandling", "InitialState '" $ string(A.InitialState) $ "' could not be set for" @ A);
		}
	}
	if (Property.Name ~= "GotoState")
	{
		A.GotoState(StringToName(Property.Value));
		if (string(A.GetStateName()) ~= Property.Value)
		{
			goto Change_Success;
			//Logger(LOG_Debug, "SpecialPropertyHandling", A @ "sent successfully to state '" $ string(A.GetStateName()) $ "'");
		}
		else
		{
			goto Change_Failed;
			//Logger(LOG_Debug, "SpecialPropertyHandling", "sending" @ A @ "to state '" $ Property.Value $ "' failed");
		}
		return true;
	}
	if (Property.Name ~= "Log")
	{
		A.Log(Property.Value);
		SuccessMessage = "log message'" $ Property.Value $ "' written to log";
		goto Change_Success;
	}
	if (Property.Name ~= "Warn")
	{
		A.Warn(Property.Value);
		SuccessMessage = "warning '" $ Property.Value $ "' written to log";
		goto Change_Success;
	}
	if (Property.Name ~= "Enable")
	{
		Logger(LOG_Debug, "SpecialPropertyHandling", "Enabling '" $ Property.Value $ "' for" @ A);
		A.Enable(StringToName(Property.Value));
		goto Change_Success; // current value of the function cannot be retrieved
	}
	if (Property.Name ~= "Disable")
	{
		Logger(LOG_Debug, "SpecialPropertyHandling", "Disabling '" $ Property.Value $ "' for" @ A);
		A.Enable(StringToName(Property.Value));
		goto Change_Success; // current value of the function cannot be retrieved
		return true;
	}
	if (Property.Name ~= "Physics")
	{
		if (Property.Value != "")
		{
			if (SetPropertyPhysics(A, Property.Value))
			{
				SuccessMessage = "successfully set Physics='" $ GetEnum(Enum'EPhysics', A.Physics) $ "' for" @ A;
				goto Change_Success;
			}
			ErrorMessage = "Physics='" $ Property.Value $ "' could not be set (current value:" @ GetEnum(Enum'EPhysics', A.Physics) $ ")";
			goto Change_Failed;
		}
		ErrorMessage = "Property 'Physics' needs a value of Enum'EPhysics' (see Actor.uc)";
		goto Change_Failed;
	}
	return false; // no special handling happened

Change_Success:
	if (LogLevel >= LOG_Debug)
	{
		if (SuccessMessage == "") // generic message
			SuccessMessage = "successfully set" @ A $ "." $ Property.Name $ "=" $ A.GetPropertyText(Property.Name);
		Logger(LOG_Debug, "SpecialPropertyHandling", SuccessMessage);
	}
	goto End;

Change_Failed:
	if (ErrorMessage == "")
		ErrorMessage = "setting property '" $ Property.Name $ "' with value '" $ Property.Value $ "' failed, current value=" $ A.GetPropertyText(Property.Name);
	Logger(LOG_Warning, "SpecialPropertyHandling", ErrorMessage);
	goto End;

End:
	return true;
}



function bool StringIsArray(string s, out string ArrayName, out int ArrayIndex) {
// *s* should have been trimmed
local int i, j;

	i = InStr(s, "[");
	if (i < 0) return false;
	Logger(LOG_Debug, "StringIsArray", "[ found at index" @ i);
	j = i + 1;
	if (j >= len(s))
		goto DecimalDigitExpected;

	while (CharIsDecDigit(Asc(Mid(s, j)))) j++;
	if (i + 1 == j) // not changed -> no digit found
		goto DecimalDigitExpected;
	if (j >= len(s))
		goto ClosingBracketExpected;
	if (Mid(s, j) != "]")
		goto ClosingBracketExpected;
	if (j + 1 < len(s))
		goto UnexpectedCharactersAtEnd;

	ArrayName = left(s, i);
	ArrayIndex = int(Mid(s, i +1, j - i));
	Logger(LOG_Debug, "StringIsArray", s @ "detected as array: ArrayName='" $ ArrayName $ "', ArrayIndex=" $ ArrayIndex);
	return true;

UnexpectedCharactersAtEnd:
	Logger(LOG_Warning, "StringIsArray", "Unexpected character(s) behind closing '] in '" $ s $ "'");
	return false;

ClosingBracketExpected:
	Logger(LOG_Warning, "StringIsArray", "closing bracket expected after digits in '" $ s $ "'");
	return false;

DecimalDigitExpected:
	Logger(LOG_Warning, "StringIsArray", "decimal digit expected after '[' in '" $ s $ "', j=" $ j);
	return false;
}



function bool StringStartsWithMyLevel(string s, out string NewValue) {
// Does *s* start with "MyLevel."? If so, *NewValue* contains the map name with remaining part of *s*.
	if ( ! (left(s, 8) ~= "MyLevel.")) return false;
	NewValue = string(Level.Outer.Name) $ right(s, Len(s) - 7);
	return true;
}



function bool StringIsVector(string s, out TVector AVector, out String ErrorMessage) {
const CFunctionName = "StringIsVector";
local int i;
local string tmpS;

	if (Loglevel >= LOG_All) LoggerDirect(LOG_All, CFunctionName, "function entered with s='" $ s $ "'");
	s = Trim(s);
	if (len(s) < 3)
	{
		if (Loglevel >= LOG_Debug) LoggerDirect(LOG_Debug, CFunctionName, "len(s)=" $ Len(s) @ "is too low, must be at least 3");
		return false; // must be "(" + at least one char + ")
	}
	// first and last char must be a bracket
	if (left(s, 1) != "(" && Right(s, 1) != ")")
	{
		if (Loglevel >= LOG_Debug) LoggerDirect(LOG_Debug, CFunctionName, "no embracing '()' -> no vector");
		return false;
	}
	if (Loglevel >= LOG_All) LoggerDirect(LOG_All, CFunctionName, "embracing '()' found");
	s = Trim(Mid(s, 1, Len(s) - 2)); // remove embracing brackets
	log("s shortened to '" $ s $ "'");
	if (s == "")
	{
		ErrorMessage = "no content in embracing '()'";
		Logger(LOG_Error, CFunctionName, ErrorMessage);
		return true;
	}
	AVector.Len = 0;
	do {
		if (AVector.Len>= ArrayCount(AVector.Values))
		{
			ErrorMessage = "too many elements in vector, max." @ ArrayCount(AVector.Values) @ "are allowed";
			Logger(LOG_Error, CFunctionName, ErrorMessage);
			return true;
		}
		i = InStr(s, ",");
		if (i < 0)
		{
			if (Loglevel >= LOG_Debug) LoggerDirect(LOG_Debug, CFunctionName, "no further separator ',' found");
			// take the reminder as value:
			i = len(s);
		}
		else
			if (Loglevel >= LOG_Debug) LoggerDirect(LOG_Debug, CFunctionName, "found separator ',' at index" @ i);
		tmpS = left(s, i);
		if (tmpS == "")
		{
			ErrorMessage = "empty content in vector at vector index" @ AVector.Len;
			Logger(LOG_Error, CFunctionName, ErrorMessage);
			return true;
		}
		AVector.Values[AVector.Len] = left(s, i);
		i++;
		s = right(s, len(s) - i);
		if (Loglevel >= LOG_All) LoggerDirect(LOG_All, CFunctionName, "s shortened to '" $ s $ "'");
		if (Loglevel >= LOG_Debug) LoggerDirect(LOG_Debug, CFunctionName, "added '" $ AVector.Values[AVector.Len] $ "' at vector index" @ AVector.Len);
		if (AVector.Values[AVector.Len] == "")
		{
			ErrorMessage = "empty vector element at vector index" @ AVector.Len;
			Logger(LOG_Error, CFunctionName, ErrorMessage);
			return true;
		}
		AVector.Len++;
	} until (s == "");

	return AVector.Len != 0;
}



function name StringToName(string str) {
	SetPropertyText("NameConversionHack", str);
	return NameConversionHack;
}



event Trigger(Actor Other, Pawn EventInstigator) {
local Actor A, temp;
local int ErrorCount, ChangedCount;

	temp = Spawn(Prototype, , , Location, Rotation);
	if (temp == None)
		Logger(LOG_Error, "Trigger", "could not spawn" @ Prototype);
	else
	{
		ErrorCount = SetProperties(temp, ChangedCount);
		Logger(LOG_Info, "Trigger", ChangedCount @ "properties successful changed for" @ temp $ "," @ ErrorCount @ "errors encountered");
		if (Event != '')
			foreach AllActors(class'Actor', A, Event)
				A.Trigger(Self, Instigator);
	}
	if (bTriggerOnceOnly)
		Disable('Trigger');
}



function string Trim(string s) {
/******************************************************************************
Returns both sides timmed string *s*
******************************************************************************/
	Return TrimL(TrimR(s));
}



function string TrimL(string s) {
/******************************************************************************
Returns left side timmed string *s*
******************************************************************************/
local int i;

	while (i < len(s) && CharIsWhite(Asc(Mid(s, i, 1))))
		i++;
	return Right(s, Len(s) - i);
}



function string TrimR(string s) {
/******************************************************************************
Returns right side timmed string *s*
******************************************************************************/
local int i;

	i = Len(s);
	while (i > 0 && CharIsWhite(Asc(Mid(s, i - 1, 1))))
		i--;
	return Left(s, i);
}



function vector TVectorToVector(TVector AVector) {
local vector V;

	V.x = float(AVector.Values[0]);
	V.y = float(AVector.Values[1]);
	V.z = float(AVector.Values[2]);
	return V;
}


/* not used atm
function bool VectorsEqual(vector NumVector, TVector StringVector) {

	if (NumVector.x != float(StringVector.Values[0])) return false;
	if (NumVector.y != float(StringVector.Values[1])) return false;
	if (NumVector.z != float(StringVector.Values[2])) return false;
	return true;
}
*/





defaultproperties {
	LogLevel=LOG_Info
	bDirectional=true
	Texture=Texture'SBTSP'
}